<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Transaction; // استدعاء نموذج المعاملات
use App\Models\WithdrawalRequest; // للوصول للمستخدم الحالي
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class UserController extends Controller
{
    /**
     * جلب سجل المعاملات الخاص بالمستخدم المسجل دخوله.
     */
    public function transactions(Request $request)
    {
        // الحصول على المستخدم المسجل دخوله
        $user = $request->user();

        // جلب معاملاته مع ترقيم الصفحات
        $transactions = $user->transactions()->latest()->paginate(20); // latest() لترتيبها من الأحدث للأقدم

        // إعادة المعاملات كـ JSON
        return response()->json($transactions);
    }

    public function requestWithdrawal(Request $request)
    {
        // التحقق من صحة البيانات القادمة
        $validated = $request->validate([
            'amount' => 'required|numeric|min:1',
            'shams_cash_iban' => 'required|string|max:255',
        ]);

        $user = $request->user();
        $amount = $validated['amount'];

        // التحقق مما إذا كان رصيد المستخدم كافيًا
        if ($user->wallet->balance < $amount) {
            return response()->json(['message' => 'رصيدك غير كافٍ لإتمام عملية السحب'], 422);
        }

        // ==========================================================
        //  الإصلاح الرئيسي هنا: استخدام Transaction لضمان تنفيذ العمليتين معًا
        // ==========================================================
        try {
            DB::beginTransaction();

            // --- العملية الأولى: إنشاء طلب السحب ---
            $withdrawalRequest = WithdrawalRequest::create([
                'user_id' => $user->id,
                'amount' => $amount,
                'shams_cash_iban' => $validated['shams_cash_iban'],
                'status' => 'pending', // الحالة الافتراضية هي "قيد المراجعة"
            ]);

            // --- العملية الثانية: إنشاء سجل معاملة مطابق ---
            $user->transactions()->create([
                'type' => 'withdrawal',
                'amount' => $amount,
                'description' => 'طلب سحب قيد المراجعة',
                'related_type' => 'App\Models\WithdrawalRequest', // ربط المعاملة بطلب السحب
                'related_id' => $withdrawalRequest->id,
            ]);

            // --- العملية الثالثة (مهمة): خصم المبلغ من رصيد المستخدم فورًا ---
            $user->wallet->decrement('balance', $amount);

            DB::commit(); // إذا نجحت كل العمليات، قم بتأكيدها

            return response()->json(['message' => 'تم إرسال طلب السحب بنجاح، وتم خصم المبلغ من رصيدك'], 201);

        } catch (\Exception $e) {
            DB::rollBack(); // إذا فشلت أي عملية، تراجع عن كل شيء

            // يمكنك تسجيل الخطأ هنا للمراجعة لاحقًا
            // Log::error($e->getMessage());
            return response()->json(['message' => 'حدث خطأ غير متوقع أثناء معالجة طلبك'], 500);
        }
    }

    public function getAdminCategories()
    {
        // لقد قمنا بتغيير اسم الدالة ليتوافق مع المسار
        $categories = \App\Models\Category::select(['id', 'name'])->get();

        return response()->json(['data' => $categories]);
    }

    /**
     * جلب جميع الدول للمشرف.
     */
    public function getAdminCountries()
    {
        $countries = \App\Models\StoreProfile::query()
            ->whereNotNull('country')
            ->where('country', '!=', '')
            ->distinct()
            ->pluck('country');

        return response()->json($countries);
    }

    public function updateProfile(Request $request)
    {
        // التحقق من صحة البيانات المدخلة
        $validatedData = $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'phone' => 'sometimes|required|string|max:20',
        ]);

        // الحصول على المستخدم المسجل دخوله حاليًا
        $user = Auth::user(); // استخدام Auth::user() هو الطريقة الأكثر موثوقية

        if (! $user) {
            return response()->json(['message' => 'User not authenticated'], 401);
        }

        // تحديث بيانات المستخدم فقط إذا تم إرسالها في الطلب
        if ($request->has('name')) {
            $user->name = $validatedData['name'];
        }
        if ($request->has('phone')) {
            $user->phone = $validatedData['phone'];
        }

        $user->save();

        // إرجاع بيانات المستخدم المحدثة مع المحفظة
        return response()->json([
            'message' => 'Profile updated successfully!',
            'user' => $user->load('wallet'),
        ]);
    }

    public function updateStore(Request $request, User $user)
    {
        if ($user->role !== 'store') {
            return response()->json(['message' => 'This user is not a store.'], 400);
        }

        $validatedData = $request->validate([
            'name' => 'sometimes|required|string|max:255',
            // نجعل البريد الإلكتروني والهاتف فريدين باستثناء المستخدم الحالي
            'email' => 'sometimes|required|string|email|max:255|unique:users,email,'.$user->id,
            'phone' => 'sometimes|required|string|unique:users,phone,'.$user->id,
            'password' => 'nullable|string|min:8', // كلمة المرور اختيارية

            'store_name' => 'sometimes|required|string|max:255',
            'description' => 'sometimes|required|string|max:1000',
            'category_id' => 'sometimes|required|exists:categories,id',
            'cashback_percentage' => 'sometimes|required|numeric|min:0|max:100',
            'address' => 'sometimes|required|string|max:255',
            'country' => 'sometimes|required|string|max:255',
            'city' => 'sometimes|required|string|max:255',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'cover' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        try {
            DB::transaction(function () use ($validatedData, $user, $request) {
                // تحديث بيانات المستخدم الأساسية
                $user->update(array_filter([
                    'name' => $validatedData['name'] ?? null,
                    'email' => $validatedData['email'] ?? null,
                    'phone' => $validatedData['phone'] ?? null,
                    'password' => isset($validatedData['password']) ? Hash::make($validatedData['password']) : null,
                ]));

                // تحديث ملف المتجر
                $storeProfileData = array_filter([
                    'store_name' => $validatedData['store_name'] ?? null,
                    'description' => $validatedData['description'] ?? null,
                    'category_id' => $validatedData['category_id'] ?? null,
                    'cashback_percentage' => $validatedData['cashback_percentage'] ?? null,
                    'address' => $validatedData['address'] ?? null,
                    'country' => $validatedData['country'] ?? null,
                    'city' => $validatedData['city'] ?? null,
                    'latitude' => $validatedData['latitude'] ?? null,
                    'longitude' => $validatedData['longitude'] ?? null,
                ]);

                if ($request->hasFile('logo')) {
                    $storeProfileData['logo_url'] = $request->file('logo')->store('store_logos', 'public');
                }
                if ($request->hasFile('cover')) {
                    $storeProfileData['cover_url'] = $request->file('cover')->store('store_covers', 'public');
                }

                if (! empty($storeProfileData)) {
                    $user->storeProfile()->update($storeProfileData);
                }
            });

            return response()->json([
                'message' => 'Store updated successfully',
                'store' => $user->fresh()->load('storeProfile.category'), // إرجاع البيانات المحدثة
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to update store.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get the authenticated user's notifications.
     */
    public function getNotifications(Request $request)
    {
        $user = $request->user();

        // نبدأ ببناء الاستعلام الخاص بالإشعارات للمستخدم الحالي
        $notificationsQuery = $user->notifications()->latest(); // latest() لترتيبها من الأحدث للأقدم

        // (اختياري ولكن احترافي) فلترة حسب الحالة (مقروءة أو غير مقروءة)
        if ($request->has('status') && $request->status === 'unread') {
            $notificationsQuery->where('is_read', false);
        }

        // تنفيذ الاستعلام وجلب النتائج مع ترقيم الصفحات (pagination)
        $notifications = $notificationsQuery->paginate(15); // 15 إشعار في كل صفحة

        return response()->json($notifications);
    }
}
