<?php

namespace App\Http\Controllers\Api\V1\Admin; // تأكد من أن هذا هو الـ namespace الصحيح

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\User; // **مهم: استيراد موديل User**
use Illuminate\Http\Request;

class StoreReportController extends Controller // تأكد من أن هذا هو اسم الكلاس الصحيح
{
    /**
     * Get comprehensive report for a specific store by ID.
     *
     * @param \Illuminate\Http\Request $request
     * @param \App\Models\User $user The store user model injected from the route.
     * @return \Illuminate\Http\JsonResponse
     */
    // ==========================================================
    // **الإصلاح الحاسم هنا: قبول المتجر من الرابط مباشرة**
    // ==========================================================
    public function getStoreReport(Request $request, User $user)
    {
        // الآن، المتغير $user هو المتجر الذي تم جلبه من الرابط باستخدام الـ ID
        // لم نعد بحاجة إلى Auth::user() هنا

        // التحقق من أن المستخدم الذي تم جلبه هو متجر
        if ($user->role !== 'store') {
            return response()->json(['message' => 'Unauthorized. The requested user is not a store.'], 403);
        }

        // --- حساب الإحصائيات ---
        $invoicesQuery = Invoice::where('store_id', $user->id);

        // فلترة حسب التاريخ إذا تم توفيرها
        if ($request->has('start_date') && $request->has('end_date')) {
            $invoicesQuery->whereBetween('created_at', [$request->start_date, $request->end_date]);
        }

        $totalCashbackGranted = $invoicesQuery->sum('cashback_amount');
        $uniqueCustomers = $invoicesQuery->distinct('user_id')->count('user_id');
        $totalTransactions = $invoicesQuery->count();
        $averageCashback = $totalTransactions > 0 ? $totalCashbackGranted / $totalTransactions : 0;

        $stats = [
            'totalCashbackGranted' => (float) $totalCashbackGranted,
            'uniqueCustomers' => (int) $uniqueCustomers,
            'totalTransactions' => (int) $totalTransactions,
            'averageCashback' => (float) $averageCashback,
        ];

        // --- جلب المعاملات (الفواتير) ---
        $transactionsQuery = Invoice::where('store_id', $user->id)
            ->with('user:id,name,phone_number') // جلب اسم ورقم هاتف العميل
            ->orderBy('created_at', 'desc');

        // فلترة حسب التاريخ مرة أخرى للمعاملات
        if ($request->has('start_date') && $request->has('end_date')) {
            $transactionsQuery->whereBetween('created_at', [$request->start_date, $request->end_date]);
        }
        
        $transactions = $transactionsQuery->get(); // استخدام get() بدلاً من paginate() لتبسيط الأمر في التطبيق

        // جلب معلومات المتجر الأساسية
        $storeInfo = $user->load('storeProfile:user_id,store_name,cover_image_url');

        return response()->json([
            'storeInfo' => $storeInfo->storeProfile, // إرسال معلومات المتجر
            'stats' => $stats,
            'invoices' => $transactions, // تغيير الاسم إلى invoices ليتطابق مع موديل التطبيق
        ]);
    }
}
