<?php

namespace App\Http\Controllers\Api\V1\Store;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\ProfileUpdateRequest;
use App\Models\StoreProfile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Http\UploadedFile;

class ProfileController extends Controller
{
    // ==========================================================
    // **1. دالة SHOW (معدّلة لتضيف حالة الطلب وتكون آمنة)**
    // ==========================================================
    public function show()
    {
        $storeUser = Auth::user()->load('storeProfile');

        // **الإضافة الأولى المطلوبة: التحقق من وجود طلب معلق**
        $pendingRequest = $storeUser->profileUpdateRequests()->where('status', 'pending')->latest()->first();
        if ($pendingRequest) {
            // إذا وجد طلب معلق، أرجع رسالة خاصة وحالة 'pending_request'
            return response()->json([
                'status' => 'pending_request',
                'message' => 'لديك طلب تعديل قيد المراجعة حاليًا. لا يمكنك تقديم طلب جديد حتى تتم معالجة الطلب الحالي.',
                'request_date' => $pendingRequest->created_at->toDateTimeString(),
            ]);
        }

        // **الإصلاح الحاسم:** التحقق من وجود storeProfile قبل استخدامه لتجنب الأخطاء
        $storeProfile = $storeUser->storeProfile;

        $categories = Category::all();
        $countries = StoreProfile::whereNotNull('country')->distinct()->pluck('country');

        // بناء كائن JSON المدمج والمنظم (بنفس الشكل الذي أرسلته تمامًا)
        return response()->json([
            'status' => 'ready', // **إضافة:** إخبار التطبيق بأنه جاهز لعرض الفورم
            'profile' => [
                'id' => $storeUser->id,
                'name' => $storeUser->name,
                'email' => $storeUser->email,
                'description' => $storeProfile->description ?? null,
                'address' => $storeProfile->address ?? null,
                'phone' => $storeUser->phone ?? null,
                'cashback_percentage' => $storeProfile->cashback_percentage_all ?? 0.0,
                'category_id' => $storeProfile->category_id ?? null,
                'country' => $storeProfile->country ?? null,
                'city' => $storeProfile->city ?? null,
                'latitude' => $storeProfile->latitude ?? null,
                'longitude' => $storeProfile->longitude ?? null,
                // **تصحيح:** استخدام المسار النسبي بأمان من قاعدة البيانات
                'profile_image_url' => $storeProfile->profile_image ?? null,
                'cover_image_url' => $storeProfile->cover_image ?? null,
            ],
            'options' => [
                'categories' => $categories,
                'countries' => $countries,
            ],
        ]);
    }

    // ==========================================================
    // **2. دالة REQUEST UPDATE (معدّلة لتمنع الطلبات المتكررة)**
    // ==========================================================
    public function requestUpdate(Request $request)
    {
        $store = Auth::user();

        // **الإضافة الثانية المطلوبة: التحقق مرة أخرى من وجود طلب معلق قبل المتابعة**
        if ($store->profileUpdateRequests()->where('status', 'pending')->exists()) {
            return response()->json(['message' => 'لا يمكنك إرسال طلب تعديل جديد بينما لديك طلب آخر قيد المراجعة.'], 409); // 409 Conflict
        }

        // التحقق من جميع الحقول الممكنة (بنفس الشكل الذي أرسلته)
        $validatedData = $request->validate([
            'name' => 'sometimes|string|max:191',
            'email' => 'sometimes|email|max:191|unique:users,email,' . $store->id,
            'description' => 'nullable|string',
            'address' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:20',
            'cashback_percentage' => 'sometimes|numeric|min:0|max:100',
            'category_id' => 'sometimes|exists:categories,id',
            'country' => 'nullable|string|max:191',
            'city' => 'nullable|string|max:191',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'profile_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'cover_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'remove_profile_image' => 'nullable|boolean',
            'remove_cover_image' => 'nullable|boolean',
        ]);

        $pendingData = array_filter($validatedData);

        // **تصحيح:** التعامل مع رفع الصور وإضافتها إلى pendingData
        if ($request->hasFile('profile_image')) {
            $pendingData['profile_image'] = $this->uploadAndCopyImage($request, 'profile_image', 'logos');
        }
        if ($request->hasFile('cover_image')) {
            $pendingData['cover_image'] = $this->uploadAndCopyImage($request, 'cover_image', 'covers');
        }

        // إنشاء طلب التحديث مع جميع البيانات
        ProfileUpdateRequest::create([
            'store_id' => $store->id,
            'pending_data' => $pendingData,
            'status' => 'pending',
        ]);

        return response()->json(['message' => 'Update request submitted successfully.'], 201);
    }

    private function uploadAndCopyImage(Request $request, string $fileKey, string $subFolder): ?string
    {
        if (!$request->hasFile($fileKey) || !$request->file($fileKey)->isValid()) {
            return null;
        }

        /** @var UploadedFile $file */
        $file = $request->file($fileKey);
        $storeSlug = Str::slug(Auth::user()->name);
        
        $path = "stores/{$storeSlug}/{$subFolder}";

        // الخطوة 1: التخزين في storage/app/public
        $storagePath = $file->store($path, 'public');

        // الخطوة 2: النسخ إلى public/storage
        try {
            $sourceFilePath = Storage::disk('public')->path($storagePath);
            // **تصحيح المسار:** يجب أن يكون المسار الوجهة داخل public/storage
            $publicDestinationPath = public_path('storage/' . $storagePath);
            File::ensureDirectoryExists(dirname($publicDestinationPath));
            File::copy($sourceFilePath, $publicDestinationPath);
        } catch (\Exception $e) {
            Log::error("Failed to copy image to public folder: " . $e->getMessage());
            // لا تقم بإيقاف العملية، فقط سجل الخطأ
        }

        return $storagePath;
    }
}
