<?php

namespace App\Http\Controllers\Api\V1\Store;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Invoice;
use App\Models\Product;
use App\Models\ProfileUpdateRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB; // **مهم:** لاستخدام دوال التجميع بكفاءة
use Carbon\Carbon; 

class ProductController extends Controller
{
        // --- الدوال المساعدة للصور ---
    protected function uploadImage(Request $request, string $productName): ?string
    {
        if (!$request->hasFile('product_image') || !$request->file('product_image')->isValid()) {
            return null;
        }
        $request->validate(['product_image' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048']);
        /** @var UploadedFile $file */
        $file = $request->file('product_image');
        $storageFolderName = 'products/' . \Illuminate\Support\Str::slug($productName) . '-' . time();
        $storagePath = $file->store($storageFolderName, 'public');
        $sourceFilePath = Storage::disk('public')->path($storagePath);
        $publicDestinationPath = public_path($storagePath);
        File::ensureDirectoryExists(dirname($publicDestinationPath));
        File::copy($sourceFilePath, $publicDestinationPath);
        return $storagePath;
    }

    protected function deleteImage(?string $imagePath): void
    {
        if ($imagePath) {
            Storage::disk('public')->delete($imagePath);
            $publicPath = public_path($imagePath);
            if (File::exists($publicPath)) {
                File::delete($publicPath);
            }
        }
    }

    // --- الدوال الرئيسية ---

    public function index()
    {
        $storeId = Auth::id();
        $products = Product::where('store_id', $storeId)->latest()->get();
        return response()->json($products);
    }

    public function store(Request $request)
    {
        $store = Auth::user();
        $validatedData = $request->validate([
            'name' => 'required|string|max:191',
            'description' => 'nullable|string',
            'cashback_percentage' => 'required|numeric|min:0',
            'product_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);
        $imagePath = $this->uploadImage($request, $validatedData['name']);
        $product = $store->products()->create([
            'name' => $validatedData['name'],
            'description' => $validatedData['description'],
            'cashback_percentage' => $validatedData['cashback_percentage'],
            'product_image' => $imagePath,
            'is_active' => true,
        ]);
        return response()->json($product, 201);
    }

    // ==========================================================
    // **الدالة التي كانت مفقودة: UPDATE**
    // ==========================================================
    public function update(Request $request, Product $product)
    {
        if ($product->store_id !== Auth::id()) {
            return response()->json(['message' => 'This product does not belong to you.'], 403);
        }
        $validatedData = $request->validate([
            'name' => 'sometimes|string|max:191',
            'description' => 'nullable|string',
            'cashback_percentage' => 'sometimes|numeric|min:0',
            'product_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);
        $dataToUpdate = $request->only(['name', 'description', 'cashback_percentage']);
        if ($request->hasFile('product_image')) {
            $this->deleteImage($product->product_image);
            $imagePath = $this->uploadImage($request, $request->name ?? $product->name);
            $dataToUpdate['product_image'] = $imagePath;
        } elseif ($request->input('remove_image') == 'true') {
            $this->deleteImage($product->product_image);
            $dataToUpdate['product_image'] = null;
        }
        $product->update($dataToUpdate);
        return response()->json($product->fresh());
    }

    // ==========================================================
    // **الدالة التي كانت مفقودة: DESTROY**
    // ==========================================================
    public function destroy(Product $product)
    {
        if ($product->store_id !== Auth::id()) {
            return response()->json(['message' => 'This product does not belong to you.'], 403);
        }
        $this->deleteImage($product->product_image);
        $product->delete();
        return response()->json(['message' => 'Product deleted successfully.']);
    }

    public function getDashboardData(Request $request)
    {
        $store = Auth::user();
        $storeId = $store->id;

        // ==========================================================
        // **1. تحسين الكفاءة: حساب إحصائيات اليوم باستعلام واحد**
        // ==========================================================
        // بدلاً من جلب كل الفواتير ثم حساب المجموع، نطلب من قاعدة البيانات
        // أن تقوم بالحساب مباشرة، وهو أسرع بكثير.
        $todayStats = Invoice::where('store_id', $storeId)
            ->whereDate('created_at', Carbon::today()) // استخدام Carbon::today() للدقة
            ->select(
                DB::raw('SUM(cashback_amount) as total_cashback'),
                DB::raw('COUNT(*) as total_transactions')
            )
            ->first(); // نستخدم first() لأننا نتوقع نتيجة واحدة فقط

        // ==========================================================
        // **2. جلب آخر 10 فواتير (يبقى كما هو)**
        // ==========================================================
        $recentInvoices = Invoice::where('store_id', $storeId)
            ->with('user:id,name') // جلب بيانات العميل (الاسم فقط) مع كل فاتورة
            ->latest()
            ->take(10)
            ->get();

        // ==========================================================
        // **3. الإصلاح الحاسم والنهائي: بناء الاستجابة الكاملة**
        // ==========================================================
        // هنا نقوم بدمج كل البيانات في استجابة واحدة.
        return response()->json([
            // **أ. إضافة بيانات المستخدم (هذا يحل مشكلة "اسم غير محدد")**
            'user' => [
                'name' => $store->name,
                'email' => $store->email,
            ],
            
            // **ب. استخدام الإحصائيات المحسوبة بكفاءة**
            // نستخدم ?? 0 لضمان أن القيمة تكون صفرًا إذا لم تكن هناك فواتير اليوم.
            'total_cashback_today' => $todayStats->total_cashback ?? 0,
            'total_transactions_today' => $todayStats->total_transactions ?? 0,
            
            // **ج. إرسال الفواتير الأخيرة**
            'recent_invoices' => $recentInvoices,
        ]);
    }


    public function updateAllProductsCashback(Request $request)
    {
        $validated = $request->validate([
            'cashback_percentage' => 'required|numeric|min:0|max:100',
        ]);

        $storeId = Auth::id();

        // تحديث كل المنتجات التابعة لهذا المتجر دفعة واحدة
        \App\Models\Product::where('store_id', $storeId)
            ->update(['cashback_percentage' => $validated['cashback_percentage']]);

        return response()->json(['message' => 'Cashback percentage for all products has been updated.']);
    }
}
