<?php

namespace App\Http\Controllers\Api\V1\Store;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\Notification;
use App\Models\Product;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage; // **مهم: إضافة هذا الاستيراد**

class CashbackController extends Controller
{
    public function processCashback(Request $request)
    {
        // 1. التحقق من صحة البيانات (لا تغيير هنا)
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|string|exists:users,id',
            'items' => 'required|json',
            'invoice_image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:4096',
        ]);

        if ($validator->fails()) {
            return response()->json(['message' => 'Validation Error', 'errors' => $validator->errors()], 422);
        }

        $items = json_decode($request->items, true);
        $productIds = array_column($items, 'product_id');
        $store = Auth::user();
        $storeProductCount = Product::where('store_id', $store->id)->whereIn('id', $productIds)->count();
        if ($storeProductCount !== count($productIds)) {
            return response()->json(['message' => 'One or more products do not belong to this store.'], 403);
        }

        DB::beginTransaction();

        try {
            $customer = User::with('referrer')->find($request->customer_id);
            $products = Product::whereIn('id', $productIds)->get()->keyBy('id');

            // 2. حساب إجمالي الفاتورة والكاش باك (لا تغيير هنا)
            $totalInvoiceAmount = 0;
            $totalCashbackAmount = 0;
            foreach ($items as $item) {
                $product = $products[$item['product_id']];
                $price = (float) $item['price'];
                $totalInvoiceAmount += $price;
                $totalCashbackAmount += $price * ($product->cashback_percentage / 100.0);
            }
            $effectiveAveragePercentage = ($totalInvoiceAmount > 0) ? ($totalCashbackAmount / $totalInvoiceAmount) * 100 : 0;

            // ==========================================================
            // **3. التعديل الحاسم هنا: تطبيق نظام النسخ اليدوي المزدوج**
            // ==========================================================
            $imageUrl = null;
            if ($request->hasFile('invoice_image')) {
                $file = $request->file('invoice_image');
                // إنشاء اسم فريد للملف لتجنب الكتابة فوق الملفات القديمة
                $fileName = 'invoices/' . time() . '_' . $file->getClientOriginalName();

                // **النسخة الأولى: حفظ في storage/app/public**
                Storage::disk('public')->put($fileName, file_get_contents($file));

                // **النسخة الثانية: حفظ في public/storage**
                $publicPath = public_path('storage/' . $fileName);
                // التأكد من وجود المجلد
                if (!file_exists(dirname($publicPath))) {
                    mkdir(dirname($publicPath), 0775, true);
                }
                // نسخ الملف
                copy($file->getRealPath(), $publicPath);

                // **حفظ المسار النسبي الصحيح في قاعدة البيانات**
                $imageUrl = '/storage/' . $fileName;
            }

            // 4. إنشاء الفاتورة مع المسار الصحيح للصورة
            $invoice = Invoice::create([
                'store_id' => $store->id,
                'user_id' => $customer->id,
                'total_amount' => $totalInvoiceAmount,
                'cashback_percentage' => $effectiveAveragePercentage,
                'cashback_amount' => $totalCashbackAmount,
                'invoice_image_url' => $imageUrl, // **استخدام المسار الصحيح**
            ]);
            $invoice->products()->attach($productIds);

            // 5. منطق توزيع الكاش باك (لا تغيير هنا)
            if ($customer->referrer) {
                $customerCashback = $totalCashbackAmount / 2;
                $referrerCommission = $totalCashbackAmount / 2;
                $referrer = $customer->referrer;

                $customer->wallet->increment('balance', $customerCashback);
                Transaction::create([
                    'user_id' => $customer->id, 'type' => 'cashback', 'amount' => $customerCashback,
                    'description' => 'كاش باك من متجر '.$store->name, 'related_id' => $invoice->id, 'related_type' => Invoice::class,
                ]);

                $referrer->wallet->increment('balance', $referrerCommission);
                Transaction::create([
                    'user_id' => $referrer->id, 'type' => 'referral_commission', 'amount' => $referrerCommission,
                    'description' => 'عمولة إحالة من العميل '.$customer->name, 'related_id' => $invoice->id, 'related_type' => Invoice::class,
                ]);
                
            } else {
                $customerCashback = $totalCashbackAmount / 2;
                $customer->wallet->increment('balance', $customerCashback);
                Transaction::create([
                    'user_id' => $customer->id, 'type' => 'cashback', 'amount' => $customerCashback,
                    'description' => 'كاش باك من متجر '.$store->name, 'related_id' => $invoice->id, 'related_type' => Invoice::class,
                ]);
            }

            DB::commit();

            return response()->json(['message' => 'تمت معالجة الكاش باك بنجاح!'], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Cashback processing failed: '.$e->getMessage(), ['exception' => $e]);
            return response()->json(['message' => 'حدث خطأ أثناء معالجة العملية.'], 500);
        }
    }
}
