<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\StoreProfile;
use App\Models\User;
use Illuminate\Http\Request;

class OfferController extends Controller
{
    /**
     * جلب قائمة المتاجر (العروض) بطريقة مبسطة ومضمونة.
     */
    public function index(Request $request)
    {
        // 1. جلب جميع المستخدمين الذين هم متاجر نشطة
        $stores = User::where('role', 'store')
            ->where('status', 'active')
            ->with(['storeProfile', 'storeProfile.category']) // تحميل البيانات المرتبطة
            ->latest()
            ->get();

        // 2. تحويل البيانات يدويًا لضمان عدم حدوث أخطاء
        // هذه هي الخطوة الأهم التي تمنع أخطاء التحويل التلقائي
        $formattedStores = $stores->map(function ($store) {
            return [
                'id' => $store->id,
                'name' => $store->name,
                'email' => $store->email,
                // إضافة البيانات من store_profile بشكل آمن
                'store_profile' => $store->storeProfile ? [
                    'category_id' => $store->storeProfile->category_id,
                    'country' => $store->storeProfile->country,
                    'city' => $store->storeProfile->city,
                    'cashback_percentage_all' => $store->storeProfile->cashback_percentage_all,
                    'profile_image_url' => $store->storeProfile->profile_image_url,
                    // إضافة بيانات الفئة بشكل آمن
                    'category' => $store->storeProfile->category ? [
                        'id' => $store->storeProfile->category->id,
                        'name' => $store->storeProfile->category->name,
                    ] : null,
                ] : null,
            ];
        });

        // 3. إرجاع البيانات بتنسيق متوافق مع Laravel Pagination
        // هذا يضمن أن تطبيق فلاتر سيستمر في العمل كما هو
        return response()->json([
            'current_page' => 1,
            'data' => $formattedStores,
            'first_page_url' => url()->current(),
            'from' => 1,
            'last_page' => 1,
            'last_page_url' => url()->current(),
            'links' => [],
            'next_page_url' => null,
            'path' => url()->current(),
            'per_page' => $stores->count(),
            'prev_page_url' => null,
            'to' => $stores->count(),
            'total' => $stores->count(),
        ]);
    }

    // ... (داخل كلاس OfferController)
    public function show($storeId) // 1. استقبال الـ ID كنص عادي
    {
        // 2. البحث عن المستخدم (المتجر) يدويًا باستخدام findOrFail
        $store = User::where('role', 'store')->findOrFail($storeId);

        // 3. تحميل البيانات اللازمة (بما في ذلك المنتجات)
        $store->load(['storeProfile.category', 'products']);

        return response()->json($store);
    }

    public function getCountries()
    {
        return response()->json(StoreProfile::query()->whereNotNull('country')->distinct()->pluck('country'));
    }

    public function getCategories()
    {
        // هذا الجزء يبقى كما هو
        $categories = \App\Models\Category::all();

        return response()->json($categories);
    }

    // ... (داخل كلاس OfferController)
    public function getCities()
    {
        // جلب جميع المدن الفريدة وغير الفارغة من جدول store_profiles
        $cities = \App\Models\StoreProfile::query()
            ->select('city')
            ->whereNotNull('city')
            ->where('city', '!=', '')
            ->distinct()
            ->pluck('city');

        return response()->json($cities);
    }
}
