<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class AuthController extends Controller
{
    // =================================================================
    // دالة التسجيل الجديدة (تم إصلاحها بالكامل)
    // =================================================================
    public function register(Request $request)
    {
        // 1. التحقق من صحة البيانات المدخلة
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'phone' => 'required|string|unique:users,phone',
            'referral_code' => [
                'required',
                'string',
                function ($attribute, $value, $fail) {
                    // الكود السحري PHOENIX مسموح به دائمًا
                    if ($value === 'PHOENIX') {
                        return;
                    }
                    // أي كود آخر يجب أن يكون موجودًا في قاعدة البيانات
                    if (! User::where('referral_code', $value)->exists()) {
                        $fail('The selected referral code is invalid.');
                    }
                },
            ],
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        // 2. تحديد المُحيل (إن وجد) - هذا هو الإصلاح المنطقي
        $referrer = null;
        if ($request->referral_code !== 'PHOENIX') {
            $referrer = User::where('referral_code', $request->referral_code)->first();
        }

        // 3. إنشاء المستخدم الجديد
        $user = User::create([
            'id' => Str::uuid(),
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'password' => Hash::make($request->password),
            // إذا وجدنا مُحيل، استخدم الـ id الخاص به، وإلا اجعله null
            'referrer_id' => $referrer ? $referrer->id : null,
            'referral_code' => 'PHOENIX-'.strtoupper(Str::random(5)),
        ]);

        // 4. إنشاء محفظة للمستخدم الجديد
        $user->wallet()->create(['balance' => 0]);

        // 5. إرجاع رسالة نجاح
        return response()->json([
            'message' => 'User registered successfully. Awaiting admin approval.',
            'user' => $user,
        ], 201);
    }

    // =================================================================
    // دالة تسجيل الدخول (لا تغييرات هنا)
    // =================================================================
    // داخل ملف: app/Http/Controllers/Api/V1/AuthController.php

    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        if (! auth()->attempt($request->only('email', 'password'))) {
            return response()->json(['message' => 'Invalid credentials'], 401);
        }

        $user = User::where('email', $request->email)->firstOrFail();

        if ($user->status !== 'active') {
            return response()->json([
                'message' => 'Your account is not active. Please contact support.',
                'status' => $user->status,
            ], 403);
        }

        $token = $user->createToken('auth_token')->plainTextToken;

        // ==========================================================
        //  **الإصلاح الحاسم والنهائي هنا: تعديل السطر الأخير فقط**
        //  نقوم بتحميل كل من 'wallet' و 'storeProfile' معًا.
        // ==========================================================
        return response()->json([
            'message' => 'Login successful',
            'access_token' => $token,
            'token_type' => 'Bearer',
            'user' => $user->load(['wallet', 'storeProfile']), // <-- **التعديل هنا**
        ]);
    }

    // =================================================================
    // دالة تسجيل الخروج (لا تغييرات هنا)
    // =================================================================
    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json(['message' => 'Successfully logged out']);
    }
}
