<?php

namespace App\Http\Controllers\Api\V1\Admin;

use App\Http\Controllers\Controller;
use App\Models\Notification;
use App\Models\WithdrawalRequest; // <-- 1. استيراد موديل Notification
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class WithdrawalController extends Controller
{
    public function index(Request $request)
    {
        $query = WithdrawalRequest::with('user:id,name,email');
        if ($request->has('status') && in_array($request->status, ['pending', 'approved', 'rejected'])) {
            $query->where('status', $request->status);
        }
        $requests = $query->oldest()->paginate(20);

        return response()->json($requests);
    }

    public function process(Request $request, WithdrawalRequest $withdrawalRequest)
    {
        $request->validate([
            'status' => ['required', Rule::in(['approved', 'rejected'])],
            'admin_notes' => 'nullable|string',
        ]);

        if ($withdrawalRequest->status !== 'pending') {
            return response()->json(['message' => 'This withdrawal request has already been processed.'], 409);
        }

        $user = $withdrawalRequest->user;
        $newStatus = $request->status;

        try {
            DB::transaction(function () use ($withdrawalRequest, $newStatus, $user, $request) {
                $notificationTitle = 'تحديث بشأن طلب السحب';
                $notificationBody = '';

                $originalTransaction = $user->transactions()
                    ->where('related_type', 'App\Models\WithdrawalRequest')
                    ->where('related_id', $withdrawalRequest->id)
                    ->first();

                if ($newStatus === 'approved') {
                    $notificationBody = 'تمت الموافقة على طلب السحب الخاص بك بمبلغ '.$withdrawalRequest->amount.'$';
                    if ($originalTransaction) {
                        $originalTransaction->update(['description' => 'سحب تمت الموافقة عليه']);
                    }
                } elseif ($newStatus === 'rejected') {
                    $notificationBody = 'تم رفض طلب السحب الخاص بك بمبلغ '.$withdrawalRequest->amount.'$';
                    $user->wallet->increment('balance', $withdrawalRequest->amount);
                    if ($originalTransaction) {
                        $originalTransaction->update(['description' => 'سحب مرفوض (تم إرجاع المبلغ)']);
                    }
                }

                $withdrawalRequest->update([
                    'status' => $newStatus,
                    'admin_notes' => $request->admin_notes,
                ]);

                // ==========================================================
                //  الإصلاح الرئيسي هنا: إنشاء الإشعار يدويًا ومباشرة
                // ==========================================================
                if ($user && $notificationBody) {
                    Notification::create([
                        'user_id' => $user->id,
                        'title' => $notificationTitle,
                        'body' => $notificationBody,
                        'type' => 'withdrawal_'.$newStatus, // نوع مخصص للإشعار
                    ]);
                }
            });
        } catch (\Exception $e) {
            return response()->json(['message' => 'An unexpected error occurred: '.$e->getMessage()], 500);
        }

        // في نهاية دالة process

        // **الإصلاح الحاسم هنا: إعادة تحميل الطلب مع بيانات المستخدم**
        $updatedRequest = $withdrawalRequest->fresh()->load('user:id,name,email');

        return response()->json([
            'message' => 'Withdrawal request has been successfully '.$newStatus.'.',
            'request' => $updatedRequest, // <-- **إرجاع الطلب المحدث مع المستخدم**
        ]);

    }
}
