<?php

namespace App\Http\Controllers\Api\V1\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class WalletController extends Controller
{
    public function adjustBalance(Request $request, User $user)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'type' => 'required|in:add,subtract',
            'reason' => 'required|string|max:255',
        ]);

        $amount = $validated['amount'];
        $type = $validated['type'];
        $reason = $validated['reason'];

        try {
            DB::transaction(function () use ($user, $amount, $type, $reason) {
                $wallet = $user->wallet;

                if ($type === 'subtract' && $wallet->balance < $amount) {
                    throw ValidationException::withMessages([
                        'amount' => 'The user does not have enough balance for this subtraction.',
                    ]);
                }

                // ==========================================================
                //  **الإصلاح الحاسم والنهائي هنا**
                // ==========================================================
                if ($type === 'add') {
                    $wallet->increment('balance', $amount);
                    $transactionType = 'admin_add'; // <-- **تسجيل النوع الصحيح**
                    $description = 'إيداع إداري: '.$reason;
                } else { // subtract
                    $wallet->decrement('balance', $amount);
                    $transactionType = 'admin_subtract'; // <-- **تسجيل النوع الصحيح**
                    $description = 'خصم إداري: '.$reason;
                }

                // إنشاء سجل المعاملة بالنوع والوصف الصحيحين
                $user->transactions()->create([
                    'amount' => $amount,
                    'type' => $transactionType, // <-- **استخدام النوع الديناميكي**
                    'description' => $description,
                    'related_type' => null, // لا يوجد موديل مرتبط
                    'related_id' => null,   // لا يوجد ID مرتبط
                ]);
            });

            // إرجاع الرصيد المحدث
            return response()->json([
                'message' => 'User balance adjusted successfully.',
                'new_balance' => $user->wallet()->first()->balance,
            ]);

        } catch (ValidationException $e) {
            return response()->json(['message' => $e->getMessage(), 'errors' => $e->errors()], 422);
        } catch (\Exception $e) {
            return response()->json(['message' => 'An error occurred during the transaction.', 'error' => $e->getMessage()], 500);
        }
    }
}
