<?php

namespace App\Http\Controllers\Api\V1\Admin;

use App\Http\Controllers\Controller;
use App\Models\ProfileUpdateRequest;
use App\Models\User; // **تأكد من وجود هذا السطر**
use Illuminate\Http\Request;

class UserReportController extends Controller
{
    /**
     * Display a report for the specified user.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    // ==========================================================
    //  **الإصلاح الحاسم هنا: إضافة Request $request إلى دالة getUserReport**
    // ==========================================================
    public function getUserReport(Request $request, User $user)
    {
        // تحميل البيانات الأساسية
        $user->load('referrer', 'referrals');

        // بناء استعلام المعاملات
        $transactionsQuery = $user->transactions()->latest();

        // فلترة المعاملات حسب التاريخ
        if ($request->has('start_date') && $request->has('end_date')) {
            $startDate = $request->input('start_date');
            $endDate = $request->input('end_date');
            // إضافة يوم واحد للنهاية لتشمل اليوم بأكمله
            $transactionsQuery->whereBetween('created_at', [$startDate, date('Y-m-d', strtotime($endDate.' +1 day'))]);
        }

        // تنفيذ الاستعلام مع pagination
        $transactions = $transactionsQuery->paginate(20);

        // إرجاع البيانات كـ JSON بالتنسيق الذي يتوقعه فلاتر
        return response()->json([
            'userInfo' => $user,
            'transactions' => $transactions->items(), // إرسال المعاملات كمصفوفة
            'referrer' => $user->referrer,
            'referrals' => $user->referrals,
        ]);
    }

    public function getUpdateRequests()
    {
        // **استخدام الموديل والعلاقة الصحيحة**
        $requests = ProfileUpdateRequest::with('store.storeProfile')
            ->where('status', 'pending')
            ->latest()
            ->paginate(15);

        return response()->json($requests);
    }

    // دالة لمعالجة الطلب (موافقة أو رفض)
    public function processUpdateRequest(Request $request, ProfileUpdateRequest $updateRequest)
    {
        $validated = $request->validate([
            'action' => 'required|in:approve,reject',
            'admin_notes' => 'required_if:action,reject|string', // **استخدام اسم الحقل الصحيح**
        ]);

        if ($updateRequest->status !== 'pending') {
            return response()->json(['message' => 'This request has already been processed.'], 409);
        }

        if ($validated['action'] === 'approve') {
            // إذا تمت الموافقة، قم بتحديث بيانات المتجر الحقيقية
            $storeProfile = $updateRequest->store->storeProfile;
            // **استخدام pending_data للتحديث**
            $storeProfile->update($updateRequest->pending_data);

            $updateRequest->status = 'approved';
        } else {
            $updateRequest->status = 'rejected';
            // **استخدام admin_notes لحفظ الملاحظات**
            $updateRequest->admin_notes = $validated['admin_notes'];
        }

        $updateRequest->save();

        // TODO: إرسال إشعار للمتجر بنتيجة الطلب

        return response()->json([
            'message' => "Request has been successfully {$updateRequest->status}.",
            'request' => $updateRequest,
        ]);
    }
}
