<?php

namespace App\Http\Controllers\Api\V1\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\ProfileUpdateRequest;
use App\Models\StoreProfile;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage; // **الإضافة 1: استيراد Validator**
use Illuminate\Support\Facades\Validator; // **الإضافة 1: استيراد Validator**
use Illuminate\Support\Str;
use Illuminate\Validation\Rule; // <-- **تأكد من وجود هذا السطر**
use Illuminate\Validation\Rules\Password;

class UserController extends Controller
{
    // =========================================================================
    //  القسم الأول: إدارة المستخدمين العامين (Users)
    // =========================================================================

    /**
     * جلب قائمة المستخدمين (غير المتاجر) مع الفلترة.
     */
    public function adjustBalance(Request $request, User $user)
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0.01',
            'type' => ['required', Rule::in(['deposit', 'withdrawal'])],
            'description' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json(['message' => $validator->errors()->first()], 422);
        }

        $amount = $request->input('amount');
        $type = $request->input('type');
        $description = $request->input('description');

        // ==========================================================
        //  **الإصلاح الحاسم والنهائي: التحديث اليدوي للمحفظة**
        // ==========================================================

        // استخدام DB::transaction لضمان أن العمليتين (تحديث الرصيد وإنشاء سجل)
        // تنجحان معًا أو تفشلان معًا. هذا يمنع أي تناقض في البيانات.
        try {
            DB::transaction(function () use ($user, $amount, $type, $description) {
                // جلب المحفظة المرتبطة بالمستخدم
                $wallet = $user->wallet;

                if (! $wallet) {
                    // في حالة عدم وجود محفظة لسبب ما، قم بإنشائها
                    $wallet = $user->wallet()->create(['balance' => 0]);
                }

                if ($type === 'deposit') {
                    // زيادة الرصيد
                    $wallet->balance += $amount;
                    $transactionType = 'admin_add';
                } else { // type is 'withdrawal'
                    if ($wallet->balance < $amount) {
                        // إلقاء استثناء لإيقاف المعاملة وإرجاع الخطأ
                        throw new \Exception('الرصيد غير كافٍ لإتمام عملية الخصم.');
                    }
                    // خصم الرصيد
                    $wallet->balance -= $amount;
                    $transactionType = 'admin_subtract';
                }

                // حفظ الرصيد الجديد
                $wallet->save();

                // إنشاء سجل معاملة جديد يدويًا
                Transaction::create([
                    'user_id' => $user->id,
                    'type' => $transactionType,
                    'amount' => $amount,
                    'description' => $description,
                ]);
            });
        } catch (\Exception $e) {
            // إرجاع رسالة الخطأ إذا فشلت المعاملة
            return response()->json(['message' => $e->getMessage()], 422);
        }

        // إرجاع الرصيد الجديد لضمان تحديث الواجهة فورًا
        return response()->json([
            'message' => 'تم تعديل الرصيد بنجاح.',
            'new_balance' => $user->fresh()->wallet->balance,
        ]);
    }

    public function toggleUserStatus(User $user)
    {
        if ($user->id === auth()->id()) {
            return response()->json(['message' => 'لا يمكنك حظر حسابك الخاص.'], 403);
        }
        $user->status = $user->status === 'active' ? 'blocked' : 'active';
        $user->save();

        return response()->json(['new_status' => $user->status]); // **الإصلاح: إرجاع المفتاح الصحيح**
    }

    public function index(Request $request)
    {
        // **التطوير 1: البدء بالاستعلام عن المستخدمين العاديين فقط**
        $query = User::where('role', 'user')->with('wallet');

        // **التطوير 2: إضافة منطق البحث الفعال**
        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                    ->orWhere('email', 'like', "%{$searchTerm}%");
            });
        }

        // **التطوير 3: تحسين فلتر الحالة ليكون أكثر تحديدًا**
        if ($request->filled('status') && in_array($request->input('status'), ['active', 'pending', 'suspended', 'blocked'])) {
            $query->where('status', $request->input('status'));
        }

        // الكود الحالي للترتيب والترقيم يبقى كما هو
        $users = $query->latest()->paginate(20);

        return response()->json($users);
    }

    /**
     * الموافقة على مستخدم معلق.
     */
    public function approve(User $user)
    {
        if ($user->status !== 'pending') {
            return response()->json(['message' => 'This user is not in a pending state.'], 409);
        }

        $user->status = 'active';
        $user->qr_code_data = 'user_id:'.$user->id;
        $user->save();

        return response()->json(['message' => 'User approved successfully.', 'user' => $user->fresh()]);
    }

    // =========================================================================
    //  القسم الثاني: إدارة المتاجر (Stores) - **مع الفلترة القوية**
    // =========================================================================

    /**
     * **الدالة الرئيسية والمطورة لجلب المتاجر مع الفلاتر القوية.**
     */
    // في ملف app/Http/Controllers/Api/V1/Admin/UserController.php

    // في ملف app/Http/Controllers/Api/V1/Admin/UserController.php

    public function getStores(Request $request)
    {
        $query = User::where('role', 'store')->with(['storeProfile.category']);

        // --- الفلاتر (تبقى كما هي) ---
        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'like', '%'.$searchTerm.'%')
                    ->orWhereHas('storeProfile', function ($profileQuery) use ($searchTerm) {
                        $profileQuery->where('store_name', 'like', '%'.$searchTerm.'%');
                    });
            });
        }
        if ($request->filled('status') && in_array($request->input('status'), ['active', 'pending', 'blocked'])) {
            $query->where('status', $request->input('status'));
        }
        if ($request->filled('category_id')) {
            $query->whereHas('storeProfile', function ($q) use ($request) {
                $q->where('category_id', $request->input('category_id'));
            });
        }
        if ($request->filled('city')) {
            $query->whereHas('storeProfile', function ($q) use ($request) {
                $q->where('city', $request->input('city'));
            });
        }

        $stores = $query->latest()->paginate(20);

        // ==========================================================
        //  **الجزء الحاسم الذي يجب التأكد من وجوده**
        // ==========================================================
        $stores->getCollection()->transform(function ($store) {
            if ($store->storeProfile) {
                if ($store->storeProfile->profile_image) {
                    $store->storeProfile->full_profile_image_url = asset(Storage::url($store->storeProfile->profile_image));
                } else {
                    $store->storeProfile->full_profile_image_url = null;
                }
                if ($store->storeProfile->cover_image) {
                    $store->storeProfile->full_cover_image_url = asset(Storage::url($store->storeProfile->cover_image));
                } else {
                    $store->storeProfile->full_cover_image_url = null;
                }
            }

            return $store;
        });

        return response()->json($stores);
    }

    /**
     * تبديل حالة المتجر (نشط/محظور).
     */
    public function toggleStoreStatus(User $user)
    {
        if ($user->role !== 'store') {
            return response()->json(['message' => 'This user is not a store.'], 400);
        }
        $newStatus = $user->status === 'active' ? 'blocked' : 'active';
        $user->update(['status' => $newStatus]);

        return response()->json(['message' => 'Store status updated successfully.', 'new_status' => $newStatus]);
    }

    /**
     * إنشاء حساب متجر جديد بالكامل مع الصور.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function createStore(Request $request)
    {
        // ==========================================================
        //  الخطوة 1: التحقق من صحة البيانات (Validation) - لا تغيير هنا
        // ==========================================================
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'store_name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => ['required', 'string', Password::defaults()],
            'phone' => 'required|string|unique:users,phone',
            'description' => 'required|string',
            'cashback_percentage' => 'required|numeric|min:0|max:100',
            'category_id' => 'required|exists:categories,id',
            'country' => 'required|string|max:255',
            'city' => 'required|string|max:255',
            'address' => 'required|string|max:255',
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
            'logo' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:4096',
            'cover' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:4096',
        ]);

        // ==========================================================
        //  الخطوة 2: إنشاء المستخدم (User) - لا تغيير هنا
        // ==========================================================
        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'phone' => $validated['phone'],
            'role' => 'store',
            'status' => 'active',
            'referral_code' => 'STORE-'.strtoupper(Str::random(6)),
        ]);

        // ==========================================================
        //  الخطوة 3: التعامل مع رفع الصور (مطابق تمامًا لمنطق التعديل)
        // ==========================================================
        
        // 3.1. إنشاء اسم فريد للشعار ونقله إلى المجلد العام
        $logoName = time() . '_logo_' . $request->file('logo')->getClientOriginalName();
        $request->file('logo')->move(public_path('storage/stores/logos'), $logoName);
        $logoPath = 'stores/logos/' . $logoName;

        // 3.2. إنشاء اسم فريد للغلاف ونقله إلى المجلد العام
        $coverName = time() . '_cover_' . $request->file('cover')->getClientOriginalName();
        $request->file('cover')->move(public_path('storage/stores/covers'), $coverName);
        $coverPath = 'stores/covers/' . $coverName;

        // ==========================================================
        //  الخطوة 4: إنشاء ملف المتجر (StoreProfile) مع المسارات الصحيحة
        // ==========================================================
        $user->storeProfile()->create([
            'store_id' => $user->id,
            'store_name' => $validated['store_name'],
            'description' => $validated['description'],
            'category_id' => $validated['category_id'],
            'country' => $validated['country'],
            'city' => $validated['city'],
            'address' => $validated['address'],
            'latitude' => $validated['latitude'],
            'longitude' => $validated['longitude'],
            'cashback_percentage_all' => $validated['cashback_percentage'],
            'profile_image_url' => $logoPath, // استخدام المسار الصحيح
            'cover_image_url' => $coverPath,   // استخدام المسار الصحيح
        ]);

        // ==========================================================
        //  الخطوة 5: إرجاع استجابة ناجحة مع البيانات الكاملة
        // ==========================================================
        return response()->json($user->load('storeProfile.category'), 201);
    }

    /**
     * تحديث بيانات متجر موجود بالكامل مع الصور.
     *
     * @param Request $request
     * @param User $user
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateStore(Request $request, User $user)
    {
        // التحقق من أن المستخدم هو متجر
        if ($user->role !== 'store') {
            return response()->json(['message' => 'User is not a store.'], 404);
        }

        // ==========================================================
        //  الخطوة 1: التحقق من صحة البيانات (Validation) - لا تغيير هنا
        // ==========================================================
        $validatedData = $request->validate([
            'name' => 'sometimes|string|max:255',
            'email' => [
                'sometimes',
                'string',
                'email',
                'max:255',
                Rule::unique('users')->ignore($user->id),
            ],
            'phone' => 'nullable|string|max:20',
            'password' => 'nullable|string|min:8',
            'store_name' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'category_id' => 'sometimes|integer|exists:categories,id',
            'cashback_percentage' => 'sometimes|numeric|min:0|max:100',
            'country' => 'sometimes|string|max:255',
            'city' => 'sometimes|string|max:255',
            'address' => 'nullable|string|max:255',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:4096',
            'cover' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:4096',
        ]);

        // ==========================================================
        //  الخطوة 2: تحديث جدول 'users' - لا تغيير هنا
        // ==========================================================
        $user->name = $validatedData['name'] ?? $user->name;
        $user->email = $validatedData['email'] ?? $user->email;
        $user->phone = $validatedData['phone'] ?? $user->phone;

        if (!empty($validatedData['password'])) {
            $user->password = bcrypt($validatedData['password']);
        }
        $user->save();

        // ==========================================================
        //  الخطوة 3: تحديث أو إنشاء ملف المتجر - لا تغيير هنا
        // ==========================================================
        
        $profileData = [
            'store_name' => $validatedData['store_name'] ?? null,
            'description' => $validatedData['description'] ?? null,
            'category_id' => $validatedData['category_id'] ?? null,
            'country' => $validatedData['country'] ?? null,
            'city' => $validatedData['city'] ?? null,
            'address' => $validatedData['address'] ?? null,
            'latitude' => $validatedData['latitude'] ?? null,
            'longitude' => $validatedData['longitude'] ?? null,
        ];

        if (isset($validatedData['cashback_percentage'])) {
            $profileData['cashback_percentage_all'] = $validatedData['cashback_percentage'];
        }

        $storeProfile = $user->storeProfile()->updateOrCreate(
            ['store_id' => $user->id],
            $profileData
        );

        // ==========================================================
        //  الخطوة 4: التعامل مع رفع الصور (الحل النهائي الصحيح)
        // ==========================================================
        if ($request->hasFile('logo')) {
            // 1. حذف الصورة القديمة إذا كانت موجودة
            if ($storeProfile->profile_image_url) {
                // نحذف الصورة من مجلد public مباشرة
                if (file_exists(public_path($storeProfile->profile_image_url))) {
                    unlink(public_path($storeProfile->profile_image_url));
                }
            }
            // 2. إنشاء اسم فريد للملف
            $logoName = time() . '_' . $request->file('logo')->getClientOriginalName();
            // 3. نقل الملف مباشرة إلى مجلد 'public/storage/stores/logos'
            $request->file('logo')->move(public_path('storage/stores/logos'), $logoName);
            // 4. حفظ المسار الصحيح الذي يمكن الوصول إليه من المتصفح
            $storeProfile->profile_image_url = 'stores/logos/' . $logoName;
        }

        if ($request->hasFile('cover')) {
            // 1. حذف الصورة القديمة إذا كانت موجودة
            if ($storeProfile->cover_image_url) {
                if (file_exists(public_path($storeProfile->cover_image_url))) {
                    unlink(public_path($storeProfile->cover_image_url));
                }
            }
            // 2. إنشاء اسم فريد للملف
            $coverName = time() . '_' . $request->file('cover')->getClientOriginalName();
            // 3. نقل الملف مباشرة إلى مجلد 'public/storage/stores/covers'
            $request->file('cover')->move(public_path('storage/stores/covers'), $coverName);
            // 4. حفظ المسار الصحيح الذي يمكن الوصول إليه من المتصفح
            $storeProfile->cover_image_url = 'stores/covers/' . $coverName;
        }
        
        // حفظ تغييرات الصور (إذا حدثت)
        if ($storeProfile->isDirty(['profile_image_url', 'cover_image_url'])) {
            $storeProfile->save();
        }

        // ==========================================================
        //  الخطوة 5: إرجاع استجابة ناجحة - لا تغيير هنا
        // ==========================================================
        return response()->json([
            'message' => 'Store updated successfully!',
            'store' => $user->fresh()->load('storeProfile.category'),
        ]);
    }
    
    /**
     * حذف متجر بشكل نهائي.
     */
    public function deleteStore(User $user)
    {
        if ($user->role !== 'store') {
            return response()->json(['message' => 'This user is not a store.'], 404);
        }

        if ($user->storeProfile) {
            if ($user->storeProfile->profile_image_url) {
                Storage::delete(str_replace('/storage', '/public', $user->storeProfile->profile_image_url));
            }
            if ($user->storeProfile->cover_image_url) {
                Storage::delete(str_replace('/storage', '/public', $user->storeProfile->cover_image_url));
            }
        }
        $user->delete();

        return response()->json(['message' => 'Store has been deleted successfully.']);
    }

    // =========================================================================
    //  القسم الثالث: جلب بيانات الفلاتر
    // =========================================================================

    /**
     * جلب جميع الفئات المستخدمة في الفلاتر.
     */
    public function getFilterCategories()
    {
        $categories = Category::select(['id', 'name'])->get();

        return response()->json($categories);
    }

    /**
     * جلب جميع المدن المستخدمة في الفلاتر.
     */
    public function getFilterCities()
    {
        $cities = StoreProfile::query()
            ->whereNotNull('city')->where('city', '!=', '')
            ->distinct()->pluck('city');

        return response()->json($cities);
    }

    // =========================================================================
    //  القسم الرابع: إدارة طلبات تحديث الملفات الشخصية
    // =========================================================================

    public function getUpdateRequests()
    {
        $requests = ProfileUpdateRequest::with('store:id,name')->where('status', 'pending')->latest()->get();

        return response()->json($requests);
    }

    public function processUpdateRequest(Request $request, ProfileUpdateRequest $updateRequest)
    {
        $request->validate(['status' => 'required|in:approved,rejected', 'admin_notes' => 'nullable|string']);

        if ($updateRequest->status !== 'pending') {
            return response()->json(['message' => 'This request has already been processed.'], 409);
        }

        if ($request->status === 'approved') {
            $storeProfile = $updateRequest->store->storeProfile;
            $pendingData = json_decode($updateRequest->pending_data, true);
            $storeProfile->fill($pendingData)->save();
        }

        $updateRequest->status = $request->status;
        $updateRequest->admin_notes = $request->admin_notes;
        $updateRequest->save();

        return response()->json(['message' => 'Update request processed successfully.']);
    }

    public function getAdminCountries()
    {
        $countries = StoreProfile::query()
            ->whereNotNull('country')
            ->where('country', '!=', '')
            ->distinct()
            ->pluck('country');

        return response()->json($countries);
    }

    public function getAdminCategories()
    {
        $categories = Category::all();

        return response()->json(['data' => $categories]);
    }

    public function getStoreProducts(User $user)
    {
        // التأكد من أن المستخدم هو متجر
        if ($user->role !== 'store') {
            return response()->json(['message' => 'This user is not a store.'], 404);
        }

        // جلب المنتجات المرتبطة بالمتجر
        // نفترض أن العلاقة في موديل User اسمها 'products'
        $products = $user->products()->latest()->get();

        return response()->json($products);
    }

    /**
     * Toggle the status of a user (active/blocked).
     *
     * @return \Illuminate\Http\JsonResponse
     */
}
