<?php

namespace App\Http\Controllers\Api\V1\Admin;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class StoreReportController extends Controller
{
    /**
     * دالة جلب تقرير متجر كامل للأدمن.
     * تم تعديلها لتشمل الفواتير المحذوفة ناعمًا (Soft Deleted).
     */
    public function getStoreReport(Request $request, User $user)
    {
        if ($user->role !== 'store') {
            return response()->json(['message' => 'This user is not a store.'], 404);
        }

        // ==========================================================
        // **1. الإصلاح الرئيسي: جلب كل الفواتير بما فيها المحذوفة**
        //    نستخدم ->withTrashed() لجعل الأدمن يرى كل شيء.
        // ==========================================================
        $baseQuery = Invoice::where('store_id', $user->id)->withTrashed();

        // الفلترة بالتاريخ تبقى كما هي
        if ($request->filled('start_date')) {
            $baseQuery->whereDate('created_at', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $baseQuery->whereDate('created_at', '<=', $request->end_date);
        }

        // ==========================================================
        // **2. تعديل الإحصائيات: يجب أن تعمل على الفواتير غير المحذوفة فقط**
        //    لأننا لا نريد أن تدخل الفواتير المحذوفة في حسابات الإجمالي.
        // ==========================================================
        $activeInvoicesQuery = Invoice::where('store_id', $user->id); // استعلام جديد للإحصائيات فقط
        if ($request->filled('start_date')) {
            $activeInvoicesQuery->whereDate('created_at', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $activeInvoicesQuery->whereDate('created_at', '<=', $request->end_date);
        }

        $stats = [
            'totalCashbackGiven' => (float) $activeInvoicesQuery->clone()->sum('cashback_amount'),
            'transactionCount' => (int) $activeInvoicesQuery->clone()->count(),
            'uniqueCustomerCount' => (int) $activeInvoicesQuery->clone()->distinct('user_id')->count('user_id'),
            'averageCashbackValue' => (float) ($activeInvoicesQuery->clone()->avg('cashback_amount') ?? 0),
        ];

        // جلب قائمة الفواتير (بما فيها المحذوفة) للعرض
        $invoices = $baseQuery->clone()
            ->with(['user:id,name,phone'])
            ->latest()
            ->get();

        $storeInfo = $user->load('storeProfile.category')->storeProfile;

        return response()->json([
            'storeInfo' => $storeInfo,
            'stats' => $stats,
            'invoices' => $invoices,
        ]);
    }

    /**
     * دالة جلب تفاصيل فاتورة واحدة (تبقى كما هي).
     * يمكنها جلب الفواتير المحذوفة بفضل Route Model Binding.
     */
    public function getInvoiceDetails(Request $request, Invoice $invoice)
    {
        $invoice->load([
            'user:id,name,phone',
            'store:id,name',
            'store.storeProfile:store_id,address',
            'products:id,name',
        ]);

        return response()->json($invoice);
    }

    /**
     * دالة حذف فاتورة واحدة (الآن تقوم بالحذف الناعم تلقائيًا).
     * تم تغيير اسمها لتطابق الراوتس التي أرسلتها.
     */
    public function destroyTransaction(Request $request, Invoice $invoice)
    {
        // بفضل تفعيل SoftDeletes في موديل Invoice،
        // هذه الدالة ستقوم الآن بتعبئة حقل 'deleted_at' بدلاً من الحذف النهائي.
        $invoice->delete();
        
        return response()->json(['message' => 'Invoice soft deleted successfully.'], 200);
    }

    /**
     * دالة تصفير (حذف ناعم) لجميع فواتير متجر معين.
     */
    public function clearStoreInvoices(Request $request, User $user)
    {
        if ($user->role !== 'store') {
            return response()->json(['message' => 'This user is not a store.'], 404);
        }

        // استدعاء دالة الحذف على كل الفواتير سيقوم بالحذف الناعم لكل واحدة.
        Invoice::where('store_id', $user->id)->delete();

        return response()->json(['message' => "All invoices for store '{$user->name}' have been soft deleted."], 200);
    }
}
