<?php

namespace App\Http\Controllers\Api\V1\Admin;

use App\Http\Controllers\Controller;
use App\Models\ProfileUpdateRequest;
use App\Models\Category; // **مهم:** تأكد من وجود هذا الاستيراد
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Storage;


class ProfileUpdateRequestController extends Controller
{
    public function index()
    {
        $requests = ProfileUpdateRequest::with(['store:id,name'])
            ->latest()
            ->get();
        return response()->json($requests);
    }

    // ==========================================================
    // **الدالة التي يجب إصلاحها بالكامل: SHOW**
    // ==========================================================
    public function show(ProfileUpdateRequest $updateRequest) // **استخدام Route Model Binding**
    {
        try {
            // تحميل العلاقات اللازمة
            $updateRequest->load('store.storeProfile.category');
            $store = $updateRequest->store;

            if (!$store || !$store->storeProfile) {
                return response()->json(['message' => 'Store or store profile not found for this request.'], 404);
            }

            // **مهم:** استخدام المتغير الصحيح هنا
            $storeProfile = $store->storeProfile;

            // بناء البيانات الحالية
            $currentProfile = [
                'name' => $store->name,
                'email' => $store->email,
                'phone' => $store->phone,
                'description' => $storeProfile->description,
                'address' => $storeProfile->address,
                'city' => $storeProfile->city,
                'country' => $storeProfile->country,
                'category_name' => optional($storeProfile->category)->name,
                'latitude' => (float) $storeProfile->latitude,
                'longitude' => (float) $storeProfile->longitude,
                // ==========================================================
                // **الإصلاح الحاسم والنهائي هنا**
                // **استخدام المتغير الصحيح ($storeProfile) لقراءة مسارات الصور**
                // ==========================================================
                'profile_image_url' => $storeProfile->profile_image_url, // <-- **تم الإصلاح هنا**
                'cover_image_url' => $storeProfile->cover_image_url,   // <-- **تم الإصلاح هنا**
            ];

            $pendingData = $updateRequest->pending_data;
            if (isset($pendingData['category_id'])) {
                $pendingData['category_name'] = optional(Category::find($pendingData['category_id']))->name;
            }

            return response()->json([
                'current_profile' => $currentProfile,
                'pending_data' => $pendingData,
            ]);

        } catch (\Exception $e) {
            Log::error('Error in ProfileUpdateRequestController@show for request #' . $updateRequest->id . ': ' . $e->getMessage());
            return response()->json(['message' => 'An internal server error occurred: ' . $e->getMessage()], 500);
        }
    }

    /**
     * معالجة طلب تحديث ملف شخصي (موافقة أو رفض).
     *
     * @param Request $request
     * @param int $requestId
     * @return \Illuminate\Http\JsonResponse
     */
    public function process(Request $request, $requestId)
    {
        $validated = $request->validate([
            'action' => ['required', Rule::in(['approved', 'rejected'])],
            'admin_notes' => ['nullable', 'string', 'max:1000'],
        ]);

        $profileUpdateRequest = ProfileUpdateRequest::with('store.storeProfile')->findOrFail($requestId);

        if ($profileUpdateRequest->status !== 'pending') {
            return response()->json(['message' => 'This request has already been processed.'], 409);
        }

        DB::beginTransaction();
        try {
            if ($validated['action'] === 'approved') {
                $store = $profileUpdateRequest->store;
                $storeProfile = $store->storeProfile;
                $pendingData = $profileUpdateRequest->pending_data;

                // 1. تحديث بيانات المستخدم الأساسية (إن وجدت)
                $userData = array_intersect_key($pendingData, array_flip(['name', 'email', 'phone']));
                if (!empty($userData)) {
                    $store->update($userData);
                }

                // 2. تحديث بيانات ملف المتجر النصية والرقمية (إن وجدت)
                $profileData = array_diff_key($pendingData, $userData, array_flip(['profile_image', 'cover_image']));
                if (!empty($profileData)) {
                    $storeProfile->fill($profileData)->save();
                }

                // ==========================================================
                //  الخطوة 3: التعامل مع الصور (الحل النهائي الصحيح)
                // ==========================================================

                // 3.1. معالجة صورة الشعار (Logo)
                if (isset($pendingData['profile_image'])) {
                    $tempLogoPath = $pendingData['profile_image'];

                    // التحقق من وجود الملف المؤقت
                    if (Storage::disk('public')->exists($tempLogoPath)) {
                        // 1. حذف الصورة القديمة (الدائمة) إذا كانت موجودة
                        if ($storeProfile->profile_image_url) {
                            Storage::disk('public')->delete($storeProfile->profile_image_url);
                        }

                        // 2. إنشاء المسار الدائم الجديد
                        $newLogoPath = 'stores/logos/' . basename($tempLogoPath);

                        // 3. نقل الملف من المجلد المؤقت إلى المجلد الدائم
                        Storage::disk('public')->move($tempLogoPath, $newLogoPath);

                        // 4. تحديث قاعدة البيانات بالمسار الدائم الجديد
                        $storeProfile->profile_image_url = $newLogoPath;
                    }
                }

                // 3.2. معالجة صورة الغلاف (Cover)
                if (isset($pendingData['cover_image'])) {
                    $tempCoverPath = $pendingData['cover_image'];

                    if (Storage::disk('public')->exists($tempCoverPath)) {
                        // 1. حذف الصورة القديمة (الدائمة) إذا كانت موجودة
                        if ($storeProfile->cover_image_url) {
                            Storage::disk('public')->delete($storeProfile->cover_image_url);
                        }

                        // 2. إنشاء المسار الدائم الجديد
                        $newCoverPath = 'stores/covers/' . basename($tempCoverPath);

                        // 3. نقل الملف من المجلد المؤقت إلى المجلد الدائم
                        Storage::disk('public')->move($tempCoverPath, $newCoverPath);

                        // 4. تحديث قاعدة البيانات بالمسار الدائم الجديد
                        $storeProfile->cover_image_url = $newCoverPath;
                    }
                }

                // حفظ تغييرات الصور (إذا حدثت)
                if ($storeProfile->isDirty(['profile_image_url', 'cover_image_url'])) {
                    $storeProfile->save();
                }

            } else { // إذا كان الإجراء هو 'rejected'
                // ==========================================================
                //  الخطوة 4: حذف الصور المؤقتة في حالة الرفض
                // ==========================================================
                $pendingData = $profileUpdateRequest->pending_data;
                if (isset($pendingData['profile_image'])) {
                    Storage::disk('public')->delete($pendingData['profile_image']);
                }
                if (isset($pendingData['cover_image'])) {
                    Storage::disk('public')->delete($pendingData['cover_image']);
                }
            }

            // تحديث حالة الطلب
            $profileUpdateRequest->status = $validated['action'];
            $profileUpdateRequest->admin_notes = $validated['admin_notes'];
            $profileUpdateRequest->processed_by = auth()->id();
            $profileUpdateRequest->processed_at = now();
            $profileUpdateRequest->save();

            DB::commit();

            return response()->json(['message' => 'Request has been ' . $validated['action'] . ' successfully.']);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error processing update request #' . $requestId . ': ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
}
