<?php

namespace App\Http\Controllers\Api\V1\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\UploadedFile; // **مهم:** تأكد من إضافة هذا الاستيراد في أعلى الملف
use Illuminate\Support\Facades\File;

class AdminProductController extends Controller
{
    /**
     * دالة مساعدة لرفع الصورة وحفظ مسارها.
     */
protected function uploadImage(Request $request, string $productName): ?string
{
    // 1. التحقق من وجود ملف صالح في الطلب
    if (!$request->hasFile('product_image') || !$request->file('product_image')->isValid()) {
        return null;
    }

    // 2. التحقق من مواصفات الملف
    $request->validate([
        'product_image' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
    ]);

    /** @var UploadedFile $file */
    $file = $request->file('product_image');

    // --- الخطوة الأولى: الحفظ بالطريقة القياسية في 'storage/app/public' ---

    // أ. إنشاء مسار فريد للمجلد (للتنظيم)
    $storageFolderName = 'products/' . \Illuminate\Support\Str::slug($productName) . '-' . time();

    // ب. حفظ الصورة في قرص 'public' (الذي يشير إلى storage/app/public)
    // ستقوم دالة store بإنشاء اسم ملف فريد تلقائيًا.
    $storagePath = $file->store($storageFolderName, 'public');
    // الآن، $storagePath يحتوي على مسار مثل:
    // 'products/my-product-1678886400/aBcDeFg.jpg'

    // --- الخطوة الثانية: نسخ الملف من 'storage' إلى 'public' ---

    // أ. الحصول على المسار الكامل للملف المصدر في مجلد storage
    $sourceFilePath = Storage::disk('public')->path($storagePath);
    // الآن، $sourceFilePath يحتوي على مسار كامل على الخادم مثل:
    // '/home/user/your-project/storage/app/public/products/my-product-1678886400/aBcDeFg.jpg'

    // ب. تحديد المسار الوجهة في المجلد العام 'public'
    // سنستخدم نفس المسار النسبي ليبقى كل شيء منظمًا ومتطابقًا.
    $publicDestinationPath = public_path($storagePath);
    // الآن، $publicDestinationPath يحتوي على مسار كامل على الخادم مثل:
    // '/home/user/your-project/public/products/my-product-1678886400/aBcDeFg.jpg'

    // ج. التأكد من وجود المجلد الوجهة قبل النسخ
    // File::ensureDirectoryExists() ستقوم بإنشاء المجلدات إذا لم تكن موجودة.
    File::ensureDirectoryExists(dirname($publicDestinationPath));

    // د. نسخ الملف من المصدر (storage) إلى الوجهة (public)
    File::copy($sourceFilePath, $publicDestinationPath);

    // --- 5. إرجاع المسار الذي سيتم حفظه في قاعدة البيانات ---
    // سنرجع المسار النسبي. هذا المسار يعمل لكلا الطريقتين (مع أو بدون رابط رمزي).
    // هذا هو المسار الذي ستستخدمه في تطبيق Flutter لبناء الرابط الكامل.
    return $storagePath;
}

    /**
     * دالة مساعدة لحذف الصورة القديمة.
     */
    protected function deleteImage($imagePath)
    {
        if ($imagePath) {
            Storage::disk('public')->delete($imagePath);
        }
    }

    /**
     * إنشاء منتج جديد لمتجر معين.
     */
    public function store(Request $request, $userId)
    {
        // الخطوة 1: التحقق من وجود المستخدم (الذي هو المتجر)
        $user = User::find($userId);

        if (! $user) {
            return response()->json(['message' => 'Store (User) not found or is not a store.'], 404);
        }

        // الخطوة 2: التحقق من صحة البيانات المدخلة
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'cashback_percentage' => 'required|numeric|min:0',
            'product_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048', // إضافة التحقق من الصورة
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // الخطوة 3: رفع الصورة وحفظ مسارها
        $imagePath = $this->uploadImage($request, $request->name);

        // الخطوة 4: إنشاء المنتج وربطه بـ user_id (الذي هو store_id في جدول المنتجات)
        $product = Product::create([
            'store_id' => $user->id,
            'name' => $request->name,
            'description' => $request->description,
            'cashback_percentage' => $request->cashback_percentage,
            'product_image' => $imagePath, // <--- حفظ مسار الصورة
            'is_active' => true,
        ]);

        // الخطوة 5: إرجاع المنتج الذي تم إنشاؤه مع رسالة نجاح
        return response()->json([
            'message' => 'Product created successfully!',
            'product' => $product,
        ], 201);
    }

    /**
     * تحديث منتج موجود.
     */
    public function update(Request $request, Product $product)
    {
        $validatedData = $request->validate([
            'name' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'cashback_percentage' => 'sometimes|numeric|min:0|max:100',
            'product_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048', // إضافة التحقق من الصورة
        ]);

        $dataToUpdate = $request->only(['name', 'description', 'cashback_percentage']);

        // تحديث الصورة إذا تم إرسال صورة جديدة
        if ($request->hasFile('product_image')) {
            // 1. حذف الصورة القديمة
            $this->deleteImage($product->product_image);

            // 2. رفع الصورة الجديدة
            $imagePath = $this->uploadImage($request, $request->name ?? $product->name);
            $dataToUpdate['product_image'] = $imagePath;
        } elseif ($request->input('remove_image') === true) {
            // إذا طلب المستخدم حذف الصورة دون رفع صورة جديدة
            $this->deleteImage($product->product_image);
            $dataToUpdate['product_image'] = null;
        }

        // تحديث المنتج
        $product->update($dataToUpdate);

        return response()->json([
            'message' => 'Product updated successfully!',
            'product' => $product->fresh(), // إرجاع المنتج بالبيانات المحدثة
        ]);
    }

    /**
     * حذف منتج موجود.
     */
    public function destroy(Product $product)
    {
        // حذف الصورة من مجلد التخزين
        $this->deleteImage($product->product_image);

        // حذف المنتج من قاعدة البيانات
        $product->delete();

        return response()->json(['message' => 'Product deleted successfully.']);
    }
}