<?php

namespace App\Http\Controllers\Api\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Carbon\Carbon;
use App\Models\User; // **مهم:** تأكد من أن هذا هو المسار الصحيح لموديل User لديك

class PasswordResetController extends Controller
{
    /**
     * الدالة الأولى: طلب إرسال رمز إعادة التعيين.
     * تستقبل بريدًا إلكترونيًا، تنشئ رمزًا، وتحاول إرساله.
     */
    public function sendResetLink(Request $request)
    {
        // 1. التحقق من صحة الطلب (يجب أن يحتوي على بريد إلكتروني صالح)
        $request->validate([
            'email' => 'required|email|exists:users,email',
        ], [
            'email.exists' => 'هذا البريد الإلكتروني غير مسجل لدينا.',
        ]);

        $email = $request->email;

        // 2. حذف أي رموز قديمة لهذا البريد الإلكتروني لتجنب التضارب
        DB::table('password_reset_tokens')->where('email', $email)->delete();

        // 3. إنشاء رمز جديد
        // الرمز هو رقم عشوائي مكون من 6 أرقام
        $token = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);

        // 4. حفظ الرمز في قاعدة البيانات مع البريد الإلكتروني وتاريخ الإنشاء
        DB::table('password_reset_tokens')->insert([
            'email' => $email,
            'token' => $token, // **ملاحظة:** لا تقم بتشفير الرمز هنا، سنقارنه كنص عادي
            'created_at' => Carbon::now(),
        ]);

        // 5. إرسال البريد الإلكتروني الذي يحتوي على الرمز
        // **مهم:** يجب أن تكون قد قمت بإعداد إعدادات البريد (MAIL_*) في ملف .env
        try {
            Mail::send('emails.password_reset', ['token' => $token], function ($message) use ($email) {
                $message->to($email);
                $message->subject('رمز إعادة تعيين كلمة المرور');
            });
        } catch (\Exception $e) {
            // إذا فشل إرسال البريد، أرجع رسالة خطأ واضحة
            return response()->json([
                'message' => 'فشل إرسال البريد الإلكتروني. يرجى التحقق من إعدادات الخادم أو المحاولة لاحقًا.',
                'error' => $e->getMessage() // للتصحيح فقط
            ], 500);
        }

        // 6. إرجاع رسالة نجاح إلى تطبيق Flutter
        return response()->json([
            'message' => 'تم إرسال رمز إعادة التعيين إلى بريدك الإلكتروني بنجاح.'
        ], 200);
    }

    /**
     * الدالة الثانية: إعادة تعيين كلمة المرور باستخدام الرمز.
     * تتحقق من صحة الرمز وتقوم بتحديث كلمة المرور.
     */
    public function resetPassword(Request $request)
    {
        // 1. التحقق من صحة البيانات المدخلة
        $request->validate([
            'email' => 'required|email|exists:users,email',
            'token' => 'required|string|min:6|max:6',
            'password' => 'required|string|min:8|confirmed', // 'confirmed' تتأكد من وجود حقل 'password_confirmation' مطابق
        ]);

        // 2. البحث عن سجل إعادة التعيين في قاعدة البيانات
        $resetRecord = DB::table('password_reset_tokens')
            ->where('email', $request->email)
            ->where('token', $request->token)
            ->first();

        // 3. التحقق من وجود السجل وصلاحيته
        // إذا لم يتم العثور على السجل، فهذا يعني أن الرمز أو البريد الإلكتروني خاطئ
        if (!$resetRecord) {
            return response()->json(['message' => 'الرمز أو البريد الإلكتروني غير صالح.'], 400);
        }

        // (اختياري) التحقق من صلاحية الرمز الزمنية (مثلاً، صالح لمدة 60 دقيقة فقط)
        $expiresAt = Carbon::parse($resetRecord->created_at)->addMinutes(60);
        if (Carbon::now()->isAfter($expiresAt)) {
            // إذا انتهت صلاحية الرمز، قم بحذفه وأرجع رسالة خطأ
            DB::table('password_reset_tokens')->where('email', $request->email)->delete();
            return response()->json(['message' => 'انتهت صلاحية هذا الرمز. يرجى طلب رمز جديد.'], 400);
        }

        // 4. تحديث كلمة مرور المستخدم
        $user = User::where('email', $request->email)->first();
        $user->password = Hash::make($request->password); // تشفير كلمة المرور الجديدة
        $user->save();

        // 5. حذف الرمز من قاعدة البيانات بعد استخدامه بنجاح
        DB::table('password_reset_tokens')->where('email', $request->email)->delete();

        // 6. إرجاع رسالة نجاح إلى تطبيق Flutter
        return response()->json([
            'message' => 'تم إعادة تعيين كلمة المرور بنجاح.'
        ], 200);
    }
}
